//
// hexagon grid test :: (c)
//
// gcc -std=gnu99 -I/opt/local/include -O2 -framework OpenGL -framework GLUT -o hex hex.c
//
//

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/time.h>

// Apple Mac OS X
#ifdef __APPLE__
  #include <GLUT/glut.h>
#endif
  
// Linux systems
#ifdef __linux__
    #include <GL/glut.h>
#endif

// window size
#define DS_WIDTH 1024
#define DS_HEIGHT 768

// flags for hex position calculation
#define HEX_HOVER 1
#define HEX_CLICK 2

// map size in cells
#define MAPWIDTH 24
#define MAPHEIGHT 16



// map cell type
typedef struct {
  float r;
  float g;
  float b;
  float a;
  float h;
} mapcell;


// map data
mapcell mapdata[MAPWIDTH*MAPHEIGHT];

// screen top left offset and hex radius
float sx=33.0f, sy=33.0f;
float hr=32.0f;

// ui state variables
mapcell nextcell;
int scrolled_x=0, scrolled_y=0;
int dragstart_x=-1, dragstart_y=-1;
int dragpos_x=-1, dragpos_y=-1;
int cell_hover_x=-1, cell_hover_y=-1;


// shift the colors in a mapcell
void shiftcell(mapcell *c)
{
  float a;
  
  a=c->b;
  c->b=c->g;
  c->g=c->r;
  c->r=a;  
}


// shift a cell in the map coordinates x,y
void shift_mapcell(int x, int y)
{
  if (x >= 0 && y >= 0 && x < MAPWIDTH && y < MAPHEIGHT)
    shiftcell(&mapdata[x+y*MAPWIDTH]);
}


// shift all the neighbours of a cell at x,y
void shift_neighbours(int x, int y)
{
  if (x&1) {
    shift_mapcell(x-1, y);
    shift_mapcell(x, y-1);
    shift_mapcell(x+1, y-1);
    shift_mapcell(x+1, y);
    shift_mapcell(x+y, y+1);
    shift_mapcell(x, y-1);
  } else {
    shift_mapcell(x-1, y);
    shift_mapcell(x-1, y-1);
    shift_mapcell(x, y-1);
    shift_mapcell(x+1, y);
    shift_mapcell(x, y+1);
    shift_mapcell(x-1, y+1);
  }
}


// plot vertexes for a hex
void draw_hex(float x, float y, float r)
{
    glVertex2f(x, y-r);
    glVertex2f(x+r, y-(r/2));
    glVertex2f(x+r, y+(r/2));
    glVertex2f(x, y+r);
    glVertex2f(x-r, y+(r/2));
    glVertex2f(x-r, y-(r/2));
}


// draw a wireframe hexagon
void draw_hex_wireframe(float x, float y, float r)
{
  glBegin(GL_LINE_LOOP);
  draw_hex(x, y, r);
  glEnd();
}


// draw a filled hexagon
void draw_hex_filled(float x, float y, float r)
{
  glBegin(GL_TRIANGLE_FAN);
  glVertex2f(x, y);
  draw_hex(x, y, r);
  glVertex2f(x, y-r);
  glEnd();
}


// paint the whole hex grid on screen
void draw_hex_grid()
{
  int i, j;
  mapcell c;

  // draw map on screen. filled hex first, then wireframe outline
  for(j=0;j<MAPHEIGHT;j++) {
    for(i=0;i<MAPWIDTH;i++) {
      c=mapdata[i+j*MAPWIDTH];
      if (j&1) {
        glColor4f(c.r, c.g, c.b, c.a);
        draw_hex_filled   (sx+i*2*hr+hr - scrolled_x, sy+j*1.5*hr - scrolled_y, hr);
        glColor4f(1.0f, 1.0f, 1.0f, 0.5f);
        draw_hex_wireframe(sx+i*2*hr+hr - scrolled_x, sy+j*1.5*hr - scrolled_y, hr);
      } else {
        glColor4f(c.r, c.g, c.b, c.a);
        draw_hex_filled   (sx+i*2*hr    - scrolled_x, sy+j*1.5*hr - scrolled_y, hr);
        glColor4f(1.0f, 1.0f, 1.0f, 0.5f);
        draw_hex_wireframe(sx+i*2*hr    - scrolled_x, sy+j*1.5*hr - scrolled_y, hr);
      }
    }
  }

  // if mouse is movering on a hex, highlight it 30% white overlay and a white outline
  if (cell_hover_x >= 0 && cell_hover_y >= 0) {
    glColor4f(0.0f, 1.0f, 0.0f, 1.0f);
    if (cell_hover_y & 1) {
      glColor4f(1.0f, 1.0f, 1.0f, 0.3f);
      draw_hex_filled(sx+cell_hover_x*2*hr+hr - scrolled_x, sy+cell_hover_y*1.5*hr - scrolled_y, hr);
      glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
      draw_hex_wireframe(sx+cell_hover_x*2*hr+hr - scrolled_x, sy+cell_hover_y*1.5*hr - scrolled_y, hr);
    } else {
      glColor4f(1.0f, 1.0f, 1.0f, 0.3f);    
      draw_hex_filled(sx+cell_hover_x*2*hr    - scrolled_x, sy+cell_hover_y*1.5*hr - scrolled_y, hr);
      glColor4f(1.0f, 1.0f, 1.0f, 1.0f);    
      draw_hex_wireframe(sx+cell_hover_x*2*hr    - scrolled_x, sy+cell_hover_y*1.5*hr - scrolled_y, hr);
    }
  }  
}


// calculate if the coordinates hit a hex and set it either to hover or active
int calculate_hex(int x, int y, int set)
{
  int xpos, ypos;
  int cell_x, cell_y;
  int relative_x, relative_y;
  float a;

  cell_hover_x=-1;
  cell_hover_y=-1;

  ypos = (scrolled_y + (y - (sy-hr)));
  cell_y = ypos / (1.5*hr);
  relative_y = ypos - cell_y*(1.5*hr);
  xpos = (scrolled_x + (x - (sx-hr)));
  if (cell_y & 1) xpos-=hr;

  if (xpos >= 0 && ypos >= 0) {
    cell_x = xpos / (2*hr);
    relative_x = xpos - cell_x*(2*hr);

    if (relative_y < 0.5*hr) {
      if (relative_x < hr) {
        // left top triangle
        a=atan(
            (float)((hr/2) - (float)relative_y) / 
            (float)(relative_x)
          ) * (180.0f/3.15149f);
        if (a >= 30.0f) {
          if (!(cell_y & 1)) cell_x--;
          relative_y=(2*hr - relative_y);
          relative_x+=hr;
          cell_y--;
        }
      } else {
        // right top triangle
        a=atan(
            (float)((hr/2) - (float)relative_y) / 
            (float)((2.0f*hr) - (float)relative_x)
          ) * (180.0f/3.14159f);
        if (a >= 30.0f) {
          if (cell_y & 1) cell_x++;
          relative_y=(2*hr - relative_y);
          relative_x-=hr;
          cell_y--;

        }
      }
    }

    // if coordinates are within a cell, activate or hover it
    if (cell_x >= 0 && cell_y >= 0 && cell_x < MAPWIDTH && cell_y < MAPHEIGHT) {
      if (set == HEX_HOVER) {
        cell_hover_x=cell_x;
        cell_hover_y=cell_y;
      }
      if (set == HEX_CLICK) {
        mapdata[cell_x + cell_y*MAPWIDTH]=nextcell;
        shift_neighbours(cell_x, cell_y);
        shiftcell(&nextcell);
//        a=nextcell.b;
//        nextcell.b=nextcell.g;
//        nextcell.g=nextcell.r;
//        nextcell.r=a;
      }
    }
  }
}


// glut screen update callback
void update(int value)
{
  glutTimerFunc(20, update, value+1); // frame number in callback parameter
  glutPostRedisplay();
}
      

// glut screen paint callback
void display(void)
{
  // setup orthogonal projection and modelview matrices
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0, DS_WIDTH, 0, DS_HEIGHT, -1, 1);
  glScalef(1, -1, 1);
  glTranslatef(0, -DS_HEIGHT, 0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // clear the back buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // enable stuff
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_BLEND);  
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

  // paint hex grid on screen
  draw_hex_grid();

  // switch the back buffer to front
  glutSwapBuffers();
}


// glut mouse hover callback
void mouse_hoverfunc(int x, int y)
{
  calculate_hex(x, y, HEX_HOVER);
}


// glut mouse drag callback
void mouse_dragfunc(int x, int y)
{
  if (dragpos_x >= 0 && dragpos_y >= 0) {
    // todo: clamp movement so that map cannot be moved completely off-screen
    scrolled_x=dragstart_x + dragpos_x-x;
    scrolled_y=dragstart_y + dragpos_y-y;
  }
}


// glut mouse click callback
void mouse_clickfunc(int button, int state, int x, int y)
{
  if (state == GLUT_DOWN && button == GLUT_RIGHT_BUTTON) {
    dragpos_x=x;
    dragpos_y=y;
    dragstart_x=scrolled_x;
    dragstart_y=scrolled_y;
  }
  if (dragpos_x >= 0 && dragpos_y >= 0 && state == GLUT_UP && button == GLUT_RIGHT_BUTTON) {
    dragpos_x=-1;
    dragpos_y=-1;
  }
  if (state == GLUT_DOWN && button== GLUT_DOWN) {
    calculate_hex(x, y, HEX_CLICK);
    calculate_hex(x, y, HEX_HOVER);
  }

  // zoom
  if (button == 3) if (hr > 8.0f) hr-=1.0f;
  if (button == 4) if (hr < 96.0f) hr+=1.0f;  
}


// glut keyboard callback
void keyboardfunc(unsigned char key, int x, int y)
{
  if (key == ',') if (hr > 8.0f) hr-=1.0f;
  if (key == '.') if (hr < 96.0f) hr+=1.0f;
}


// main program
int main(int argc, char **argv)
{
  int i,j;

  // clear all map cells to black
  for(j=0;j<MAPHEIGHT;j++) {
    for(i=0;i<MAPWIDTH;i++) {  
      mapdata[i+j*MAPWIDTH].r=0.0f;
      mapdata[i+j*MAPWIDTH].g=0.0f;
      mapdata[i+j*MAPWIDTH].b=0.0f;
      mapdata[i+j*MAPWIDTH].a=1.0f;
      mapdata[i+j*MAPWIDTH].h=0.0f;
    }
  }

  // next cell color
  nextcell.r=0.6f;
  nextcell.g=0.0f;
  nextcell.b=0.0f;
  nextcell.a=1.0f;
  nextcell.h=0.0f;

  // init glut
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
  glutInitWindowSize(DS_WIDTH,DS_HEIGHT);
  glutCreateWindow("hex grid test");

  // register mouse/kb callbacks
  glutMouseFunc(mouse_clickfunc);
  glutMotionFunc(mouse_dragfunc);
  glutPassiveMotionFunc(mouse_hoverfunc);
  glutKeyboardFunc(keyboardfunc);

  // start updates
  glutDisplayFunc(display);
  glutTimerFunc(20, update, 1);

  // and go!
  glutMainLoop();
  return 0;
}
